# Chapter 9: Performance, Code Splitting, and Accessibility in React 19

# Performance, Code Splitting, and Accessibility in React 19

## Introduction: Speed, Accessibility, and the Modern Web

Imagine your app as a high-speed train. Users expect to board instantly, travel comfortably, and reach their destination—no matter their device or ability. If your train is slow or inaccessible, users will leave for a faster, friendlier ride.

Performance and accessibility are the twin engines of modern web success. Fast apps engage users and drive revenue. Accessible apps reach everyone and avoid legal risk. React 19 introduces new features—automatic optimizations, granular code splitting, and progressive enhancement—that make building fast, inclusive apps easier than ever.

In this chapter, you’ll learn:
- Why performance and accessibility matter for business
- How Core Web Vitals measure real-world user experience
- How React 19’s compiler automates performance
- How to split code and stream UI for instant feedback
- How to build apps that work for everyone, everywhere

Let’s start by understanding the metrics that define a fast, accessible app.

---

## Understanding Modern Performance Metrics

Performance is a business metric. If your React app is slow, users leave. Google’s Core Web Vitals are the industry standard for measuring real user experience.

### Core Web Vitals: What to Measure

- **Largest Contentful Paint (LCP):** How fast does main content appear? Target: under 2.5 seconds.
- **Interaction to Next Paint (INP):** How quickly does your app respond to a click or tap? Target: under 200ms.
- **Cumulative Layout Shift (CLS):** Does the page jump around as it loads? Target: below 0.1.

These metrics affect SEO and conversions. Amazon found every 100ms of delay cost 1% in sales.

#### Measuring with Lighthouse

Before you optimize, measure your app’s vitals.

### Measuring Core Web Vitals with Lighthouse

To check your app’s performance:

```sh

# 1. Open your app in Chrome
# 2. Press F12 for DevTools
# 3. Go to 'Lighthouse' tab
# 4. Click 'Analyze page load'
# 5. Review LCP, INP, and CLS scores
```

- Focus on improving the lowest scores.
- Modern hosts (Vercel, Netlify, Cloudflare) deploy React apps globally, reducing latency.

#### Performance and Revenue

Slow apps lose money. For example:

### Estimating Lost Revenue from Delay

A simple calculation:

```javascript

// If your site makes $500,000/month and LCP slows by 500ms:
const monthlyRevenue = 500000;
const lostRevenue = monthlyRevenue * 0.05; // 5% loss
console.log(`Potential monthly loss: $${lostRevenue}`); // $25,000
```

- A 500ms delay could cost $25,000/month.

#### Tracking Improvements

Track Core Web Vitals in production:

### Tracking LCP with web-vitals and Google Analytics

```javascript

import { onLCP } from 'web-vitals';
// gtag is Google Analytics’ global site tag
onLCP((metric) => {
  gtag('event', 'LCP', {
    value: metric.value,
    event_category: 'Web Vitals',
    event_label: 'Largest Contentful Paint',
    non_interaction: true
  });
});
```

- This sends LCP scores to Google Analytics.
- Use these numbers to see if optimizations work.

---

## React Compiler and Automatic Optimizations

React 19’s compiler is your automatic pit crew. It analyzes your components and optimizes them at build time. You no longer need to sprinkle `useMemo` and `useCallback` everywhere.

### Manual Memoization vs. Compiler Optimization

Before, you wrote:

```tsx

// Manual memoization
const result = useMemo(() =>
  computeValue(a, b), [a, b]);
```

With React 19, just write:

```tsx

// Compiler handles optimization
const result = computeValue(a, b);
```

- The compiler caches pure calculations.
- Only use manual memoization for impure or non-deterministic logic.

### Identifying Bottlenecks

Don’t guess—profile first. Use React DevTools Profiler:

1. Open React DevTools.
2. Go to Profiler tab.
3. Record a session.
4. Look for slow components or unnecessary re-renders.

If you find heavy work in a render, move it to a Server Component or use `useWasm` for CPU-bound logic.

---

## Component-Level Code Splitting and Lazy Loading

Shipping all your code at once is wasteful. Code splitting delivers only what users need, when they need it.

### Using React.lazy and Suspense

Load components on demand:

### Basic Code Splitting with React.lazy and Suspense

This pattern loads a heavy feature (like a product modal) only when needed.

```tsx

import React, { Suspense } from 'react';

const ProductDetails = React.lazy(() =>
  import('./ProductDetails'));

function Catalog() {
  const [showModal, setShowModal] = React.useState(false);

  return (
    <div>
      <button onClick={() => setShowModal(true)}>
        View Details
      </button>
      {showModal && (
        <Suspense fallback={
          <div>Loading product details...</div>
        }>
          <ProductDetails />
        </Suspense>
      )}
    </div>
  );
}
```

- `React.lazy` loads `ProductDetails` only when needed.
- `Suspense` shows a fallback while loading.
- Use code splitting for features not needed at first load.

### Route-Based and Component-Level Splitting

- **Route-based:** Routers like Next.js or Remix split code per page automatically.
- **Component-level:** Use `React.lazy` for features inside a page.

Modern build tools (Vite) automate chunking. Use both splitting methods for best results.

---

## Progressive Enhancement and Partial Hydration

Not every user needs all features at once. Progressive enhancement delivers core content to everyone, then adds interactivity for capable browsers.

### Server Components for Fast, Accessible Content

Render main content on the server:

### Server Component for Product List

This component ensures all users see content fast.

```tsx

// app/components/ProductList.server.tsx
import { fetchProducts } from '../lib/api';

export default async function ProductList() {
  const products = await fetchProducts();
  return (
    <ul>
      {products.map(product => (
        <li key={product.id}>{product.name}</li>
      ))}
    </ul>
  );
}
```

- Fetches products on the server.
- Renders a static HTML list for instant access.

### Partial Hydration: Hydrate Only What Matters

Hydrate only interactive islands, like buttons.

### Composing Server and Client Components

```tsx

// app/components/ProductList.server.tsx
import { fetchProducts } from '../lib/api';
import AddToCartButton from './AddToCartButton.client';

export default async function ProductList() {
  const products = await fetchProducts();
  return (
    <ul>
      {products.map(product => (
        <li key={product.id}>
          {product.name}
          <AddToCartButton productId={product.id} />
        </li>
      ))}
    </ul>
  );
}
```

- Only `AddToCartButton` is hydrated on the client.
- The rest of the list stays static and accessible.

### Client Component with Server Actions

```tsx

// app/components/AddToCartButton.client.tsx
'use client';
import { useActionState } from 'react';

export async function addToCartAction(productId) {
  const response = await fetch('/api/cart/add', {
    method: 'POST',
    body: JSON.stringify({ productId }),
    headers: { 'Content-Type': 'application/json' },
  });
  if (!response.ok) throw new Error('Failed to add');
  return { success: true };
}

export default function AddToCartButton({ productId }) {
  const [state, action, isPending] = useActionState(
    async (prev, formData) => {
      try {
        await addToCartAction(productId);
        return { added: true, error: null };
      } catch (error) {
        return { added: false, error: error.message };
      }
    },
    { added: false, error: null }
  );

  return (
    <form action={action} style={{ display: 'inline' }}>
      <button
        type="submit"
        disabled={state.added || isPending}
        aria-label="Add to cart"
      >
        {state.added
          ? 'Added!'
          : isPending
          ? 'Adding...'
          : 'Add to Cart'}
      </button>
      {state.error && (
        <span role="alert" style={{ color: 'red' }}>
          {state.error}
        </span>
      )}
    </form>
  );
}
```

- Handles server mutation and optimistic UI.
- Only the button is interactive; rest is static.

#### Best Practices

- Hydrate only interactive elements.
- Prioritize critical content.
- Test with JS disabled and on slow networks.
- Use edge deployment for global speed.
- Integrate PWA features for offline support.
- Use AI-powered tools for accessibility and performance audits.

---

## Optimizing for RSC Streaming

React Server Components (RSC) streaming sends UI as soon as each part is ready. Users see content sooner—no more blank screens.

### Enabling Streaming in Next.js 14

```tsx

// app/products/page.server.tsx
import { fetchProducts } from '../../lib/data';
import { AddToCartButton } from './AddToCartButton.client';

export default async function ProductCatalog() {
  const products = await fetchProducts();
  return (
    <ul>
      {products.map(product => (
        <li key={product.id}>
          {product.name} - ${product.price}
          <AddToCartButton productId={product.id} />
        </li>
      ))}
    </ul>
  );
}
```

- Next.js streams UI as data resolves.
- Only client components are hydrated.

### Custom Streaming with Express and Vite

```javascript

// server.js — Express entry point for RSC streaming
import { renderToPipeableStream }
  from 'react-server-dom-webpack/server';
import express from 'express';
import App from './App.server';

const app = express();

app.get('*', (req, res) => {
  const stream = renderToPipeableStream(<App />, {
    onShellReady() {
      res.statusCode = 200;
      res.setHeader('Content-Type', 'text/html');
      stream.pipe(res);
    },
    onError(error) {
      console.error(error);
      res.statusCode = 500;
      res.send('Internal Server Error');
    }
  });
});

app.listen(3000, () => {
  console.log('Server listening on http://localhost:3000');
});
```

- For most apps, prefer framework-managed streaming.

### Balancing Server and Client

- Server Components: fetch data, render static content.
- Client Components: handle interactivity only.

---

## Performance and Accessibility: Achieving Both

Performance and accessibility reinforce each other. Use semantic HTML for both speed and inclusion.

### Accessible and Performant Button Example

```tsx

<button type="submit" aria-label="Add to cart">
  Add to cart
</button>
```

- `<button>` is keyboard-accessible and recognized by screen readers.
- Avoid using `<div onClick>` for actions.

### Automated Accessibility Testing

Catch issues early with automated tools:

### Automated Accessibility Test with Vitest and jest-axe

```tsx

// __tests__/Button.a11y.test.tsx
import { render } from '@testing-library/react';
import { axe, toHaveNoViolations } from 'jest-axe';
import { expect, test } from 'vitest';
import Button from '../Button';

expect.extend(toHaveNoViolations);

test('Button is accessible', async () => {
  const { container } = render(
    <Button>Add to cart</Button>
  );
  const results = await axe(container);
  expect(results).toHaveNoViolations();
});
```

- Run these tests in CI to prevent regressions.

### WCAG Compliance

- Use semantic HTML.
- Ensure color contrast.
- Support keyboard navigation.
- Provide text alternatives.
- Test with screen readers and automated tools.

---

## Summary and Key Takeaways

Performance and accessibility are the foundation of modern React apps. Fast, inclusive experiences drive business results. React 19 automates many optimizations—focus on clean, idiomatic code and let the compiler do the heavy lifting.

- Use Core Web Vitals to measure real user experience.
- Trust React 19’s compiler for most optimizations.
- Split code by route and component for lean bundles.
- Use Server Components for static content; hydrate only interactive islands.
- Stream UI for instant feedback.
- Integrate automated accessibility and performance tests into CI/CD.

These patterns are now defaults in modern frameworks. Master them to deliver apps that are fast, accessible, and ready for enterprise scale.

---

## Key Ideas and Glossary

| Key Idea                             | Description                                              |
|--------------------------------------|----------------------------------------------------------|
| Core Web Vitals                      | LCP, INP, CLS: metrics for web performance               |
| React Compiler                       | Automates memoization and optimizations                  |
| Code Splitting                       | Loads only needed code per route/component               |
| Server Components                    | Render static content on the server                      |
| Client Components                    | Handle browser interactivity; hydrated as needed         |
| Partial Hydration                    | Hydrate only interactive parts of the UI                 |
| RSC Streaming                        | Server streams UI as soon as ready                       |
| Progressive Enhancement              | Build core for all, enhance for capable browsers         |
| Accessibility (a11y)                 | Inclusive design for all users                           |
| WCAG                                 | Web Content Accessibility Guidelines (compliance std.)   |

**Glossary:**
- **LCP:** Largest Contentful Paint. Time to show main content.
- **INP:** Interaction to Next Paint. Input response speed.
- **CLS:** Cumulative Layout Shift. Visual stability as page loads.
- **Hydration:** Making server-rendered HTML interactive in the browser.
- **Partial Hydration:** Hydrating only interactive components.
- **Suspense:** Shows fallback UI while loading lazy components.
- **PWA:** Progressive Web App. Offline-capable, installable web app.
- **Edge Deployment:** Running code close to users for global speed.
- **Axe-core, Lighthouse:** Automated accessibility/performance tools.

---

## Exercises and Next Steps

1. **Profile your React app**  
   Use Chrome DevTools or Lighthouse. List your Core Web Vitals scores and three areas to improve.

2. **Refactor for Compiler Optimization**  
   Remove unnecessary `useMemo` or `useCallback`. Measure performance and code clarity before and after.

3. **Implement Code Splitting**  
   Use `React.lazy` and `Suspense` for a feature module (modal or admin panel). Provide a loading fallback.

4. **Add Automated Accessibility Testing**  
   Integrate axe-core or Lighthouse CI into your test runner or CI scripts. Fail builds on critical accessibility issues.

5. **Build with Server Components and Partial Hydration**  
   Render main content on the server. Hydrate only interactive components. Verify the page works with JavaScript disabled.

For deeper dives, see:
- **Chapter 4:** Server Components and streaming
- **Chapter 8:** Automated testing strategies
- **Chapter 9:** Routing, performance, and accessibility
- **Chapter 11:** Real-world case study with accessible, performant UI

You’re now ready to build React apps that are fast, accessible, and future-proof.