# Chapter 5: Modern Data Flow: Actions, Mutations, and Forms in React 19

# Modern Data Flow: Actions, Mutations, and Forms in React 19

## Introduction: Data Flow Reimagined—From Paperwork to Instant Approval

Imagine applying for a loan with paper forms—slow, error-prone, and frustrating. Traditional React forms felt the same: manual state, scattered logic, and fragile validation. React 19 changes this. Now, submitting a form is like instant online approval: fast, secure, and reliable.

This chapter covers:

- **Actions:** Server-side functions for mutations (data changes).
- **Modern Form Hooks:** Hooks like `useFormAction`, `useFormStatus`, and `useActionState` for handling validation, loading, and errors.
- **Optimistic UI:** Instantly update the UI while waiting for the server.
- **Testing and Debugging:** Ensure reliability and business value.

You'll learn how to build robust, type-safe forms and mutations—no more boilerplate, no more lost data.

---

## Actions: The New Mutation API

Actions in React 19 are like digital contracts—secure, instant, and centralized. They replace manual event handlers and fetch calls with server-only functions.

### What Are Actions and Why Do They Matter?

Actions are server-side functions that handle mutations. Your UI submits data; the server validates and processes it. All sensitive logic stays server-side.

#### ### Defining a Simple Action (`createOrder.ts`)
Before the code:  
This example shows a server Action that receives form data, validates it, and returns a result.

```typescript

export async function createOrder(formData: FormData) {
  // Validate and process the order
  // ...business logic...
  return { success: true };
}
```

- Defines a server function for order creation.
- Receives a `FormData` object from the form.
- Runs all validation and business logic on the server.
- Returns a result for the UI.

**Why it matters:**  
- Centralizes business logic  
- Ensures type safety  
- Keeps sensitive checks off the client

---

### Replacing Legacy Form Handling with Server-Side Logic

Old forms required `onSubmit` handlers, fetch calls, and manual error handling. Actions simplify this—forms submit directly to the server.

#### ### Legacy Client-Side Form Submission (`LegacyOrderForm.tsx`)
Before the code:  
This example shows the traditional way—manual event handling and fetch.

```typescript

function LegacyOrderForm() {
  const handleSubmit = async (event: React.FormEvent) => {
    event.preventDefault();
    const formData = new FormData(
      event.target as HTMLFormElement
    );
    const response = await fetch('/api/orders', {
      method: 'POST',
      body: formData,
    });
    // Handle response and errors
  };
  return (
    <form onSubmit={handleSubmit}>
      {/* form fields */}
      <button type="submit">Submit</button>
    </form>
  );
}
```

- Handles form submission manually.
- Serializes form data and makes a fetch call.
- Requires manual error and loading state management.

---

### Optimistic UI and Business Workflows

Optimistic UI shows instant updates before the server responds. If the mutation fails, you roll back.

#### ### Performing an Optimistic Update with Actions (`OptimisticOrder.tsx`)
Before the code:  
This snippet shows how to update the UI optimistically when submitting a form.

```typescript

const [pending, setPending] = useState(false);

const handleAction = async (formData: FormData) => {
  setPending(true);
  try {
    await createOrder(formData);
    // Update UI optimistically
  } finally {
    setPending(false);
  }
};
```

- Sets a loading state.
- Calls the Action.
- Updates the UI before the server responds.
- Resets the loading state after completion.

---

## React 19 Form Actions and Hooks

Forms connect users to your business logic. React 19's hooks make forms reliable and easy.

### Overview of New Form-Related Hooks

- **`useFormAction`**: Connects a form or button to a server Action.
- **`useFormStatus`**: Tracks form submission status.
- **`useActionState`**: Manages Action results and errors.
- **`formAction` (button prop):** Binds different Actions to different buttons.

#### ### Using a Form Hook to Connect to an Action (`CheckoutForm.tsx`)
Before the code:  
This example connects a form to a server Action using `useFormAction`.

```typescript

function CheckoutForm() {
  const [formAction, { status, error }] =
    useFormAction(createOrder);
  return (
    <form action={formAction}>
      {/* form fields */}
      <button type="submit"
        disabled={status === 'pending'}>
        Checkout
      </button>
      {error && (
        <div role="alert">{error.message}</div>
      )}
    </form>
  );
}
```

- Connects the form to a server Action.
- Disables the button while pending.
- Shows errors accessibly.

---

### Handling Resource Loading and Form State

Show users when actions are running. Use status values for loading, success, and errors.

---

### Best Practices for Modern React Forms

- **Validate** on both client and server.
- **Use semantic HTML** and ARIA roles for accessibility.
- **Leverage TypeScript** for type safety.
- **Centralize business logic** in Actions.
- **Provide clear feedback** for all outcomes.
- **Prefer built-in APIs** over external libraries.

---

## useActionState in Practice

`useActionState` is your form's control tower—tracking submissions, errors, and feedback.

### Managing Form State and Errors

#### ### Basic useActionState Usage (`OrderForm.tsx`)
Before the code:  
This example shows a form using `useActionState` for submission and feedback.

```typescript

const [state, submitAction, isPending] =
  useActionState(createOrder, initialState);

return (
  <form action={submitAction}>
    {/* fields */}
    {state.error && <div>{state.error}</div>}
    <button disabled={isPending}>Submit</button>
  </form>
);
```

- Connects the form to a server Action.
- Tracks pending and error states.
- Disables the button during submission.

---

### Type-Safe Validation and Feedback

Define exact types for your form data and return clear errors.

#### ### Type-Safe Validation Example (`validateCheckout.ts`)
Before the code:  
This example defines a type and a validation function for checkout data.

```typescript

interface CheckoutData {
  email: string;
  cardNumber: string;
}

function validate(data: CheckoutData): string | null {
  if (!data.email.includes("@"))
    return "Invalid email address.";
  if (data.cardNumber.length < 16)
    return "Card number is too short.";
  return null;
}
```

- Defines data shape with TypeScript.
- Validates fields and returns user-friendly errors.

---

### Practical Example: Checkout Form with Payment Integration

#### ### Complete Checkout Form Using useActionState (`CheckoutForm.tsx`)
Before the code:  
This example combines validation, state, and feedback for a payment form.

```typescript

// Action
export async function processCheckout(
  formData: FormData
) {
  const email = formData.get("email") as string;
  const card = formData.get("cardNumber") as string;
  if (!email.includes("@"))
    return { error: "Invalid email address." };
  if (card.length < 16)
    return { error: "Card number is too short." };
  // ...process payment...
  return { success: true };
}

// Component
const [state, submit, pending] =
  useActionState(processCheckout, {});
return (
  <form action={submit}>
    <input name="email" type="email" required />
    <input name="cardNumber" type="text"
      required minLength={16} />
    {state.error && (
      <div role="alert">{state.error}</div>
    )}
    <button disabled={pending}>Pay Now</button>
  </form>
);
```

- Validates input server-side.
- Connects form to Action with `useActionState`.
- Shows errors and disables button while pending.

---

## Patterns for Real-World Mutations

Real apps face async operations, race conditions, and external APIs. React 19's Actions and hooks keep things reliable.

### Handling Asynchronous Operations and Race Conditions

Prevent duplicate submissions and keep the UI current.

#### ### Preventing Duplicate Submissions (`PreventDuplicate.tsx`)
Before the code:  
This pattern ensures only one submission at a time.

```typescript

const [pending, setPending] = useState(false);

const handleSubmit = async (formData: FormData) => {
  if (pending) return;
  setPending(true);
  try {
    await processCheckout(formData);
  } finally {
    setPending(false);
  }
};
```

- Checks if already pending.
- Prevents double submit.
- Resets state after completion.

---

### Integrating with External APIs Securely

Keep secrets server-side and handle API errors gracefully.

#### ### Calling an External Payment API from an Action (`processPayment.ts`)
Before the code:  
This example shows a server Action making a secure external API call.

```typescript

export async function processPayment(
  formData: FormData
) {
  try {
    const response = await fetch(
      "https://payments.example.com/api/pay",
      {
        method: "POST",
        body: formData
      }
    );
    if (!response.ok)
      throw new Error("Payment failed");
    return { success: true };
  } catch (error) {
    return { error: (error as Error).message };
  }
}
```

- Makes API call server-side.
- Catches and returns errors.
- Never exposes secrets to the client.

---

### Error Boundaries and User Experience

Error boundaries catch unexpected errors and show fallback UI.

#### ### Simple Error Boundary Component (`ErrorBoundary.tsx`)
Before the code:  
This class component catches rendering errors and displays a fallback.

```typescript

class ErrorBoundary extends React.Component {
  state = { hasError: false };
  static getDerivedStateFromError(error: unknown) {
    return { hasError: true };
  }
  render() {
    if (this.state.hasError) {
      return (
        <h2>
          Something went wrong. Please try again.
        </h2>
      );
    }
    return this.props.children;
  }
}
```

- Catches errors in child components.
- Shows a fallback message.
- Keeps the rest of the app running.

---

## Testing and Debugging Actions

Testing and debugging ensure reliability and reduce business risk.

### Writing Tests for Actions and Mutations

Test Actions directly with Vitest. Use integration tests for end-to-end flows.

#### ### Vitest Test for an Action (`processCheckout.test.ts`)
Before the code:  
This test checks that invalid input returns an error.

```typescript

import { describe, it, expect } from 'vitest';
import { processCheckout } from './actions';

describe('processCheckout', () => {
  it('returns error for invalid input', async () => {
    const formData = new FormData();
    formData.append('email', 'not-an-email');
    const result = await processCheckout(formData);
    expect(result.error).toBeDefined();
  });
});
```

- Calls the Action with invalid data.
- Asserts that an error is returned.

---

### Debugging Failed Actions and Error Propagation

Log errors server-side and show clear feedback in the UI.

---

### Business Impact: Reducing Failed Transactions and Support Tickets

Well-tested Actions reduce bugs, failed transactions, and support costs. Clear errors build user trust.

---

### See Also: Testing Strategies (Chapter 8)

For advanced testing, see Chapter 8: Testing for Confidence.

---

## Summary, Key Ideas, and Glossary

React 19+ transforms forms and mutations. Actions centralize logic, form hooks manage state and feedback, and optimistic UI keeps users happy. Testing and error boundaries ensure reliability.

### Key Ideas

- **Actions:** Server-side mutation logic, secure and centralized.
- **Form Hooks:** Handle state, validation, and feedback with less code.
- **Type Safety:** Prevents bugs early.
- **Optimistic UI:** Instant feedback, fewer delays.
- **Error Boundaries:** Resilient UI, user-friendly errors.
- **Testing:** Fewer bugs, safer releases.

### Glossary

| Term              | Definition                                                    |
|-------------------|---------------------------------------------------------------|
| **Action**        | Server-side function for handling data changes (mutations).   |
| **useActionState**| Hook for managing Action state (pending, error, success).     |
| **useOptimistic** | Hook for optimistic UI updates before server confirms change. |
| **formAction**    | Prop on `<button>` to bind to a specific Action.              |
| **Optimistic UI** | UI updates before server confirmation for better UX.          |
| **Error Boundary**| React class component that catches UI errors.                 |
| **Form Hook**     | Hook for managing form state, validation, and submission.     |

---

## Connecting to the Bigger Picture

Actions and form hooks are the backbone of robust React 19+ apps. Next, explore state management in the server-first era (Chapter 6). For deep testing strategies, see Chapter 8.

---

## Exercises and Next Steps

**Exercise 1:**  
Refactor a legacy form using `onSubmit` and fetch to use an Action and `useActionState`. Ensure type safety and error handling.

**Exercise 2:**  
Implement optimistic UI for an 'Add to Cart' button. Show instant feedback while the mutation is pending.

**Exercise 3:**  
Write a unit test for a payment Action that returns an error for invalid card details.

**Exercise 4:**  
Integrate an external API into an Action. Handle API errors and display user-friendly messages.

**Exercise 5:**  
Add an Error Boundary to your checkout page. Simulate a failure and confirm the fallback UI appears.

---

Ready to build seamless, reliable data flows? Let’s keep going—one robust mutation at a time.